import React, { useState, useEffect, useMemo } from 'react';
import { jsPDF } from 'jspdf';
import autoTable from 'jspdf-autotable';

import { ConfigState, PriceResult } from './types';
import { calculatePrice, formatCurrency } from './utils';
import { PRICES, WATER_BOILER_OPTIONS, WATER_FLOW_OPTIONS } from './constants';

import SectionCard from './components/SectionCard';
import CheckboxCard from './components/CheckboxCard';
import LeadModal from './components/LeadModal';
import Navigation from './components/Navigation';
import SummarySidebar from './components/SummarySidebar';
import MobileActionbar from './components/MobileActionbar';
import ChatWidget from './components/ChatWidget';

// --- HELPER COMPONENTS ---

const Tooltip = ({ text }: { text: string }) => (
  <div className="group relative inline-block ml-1 align-middle">
    <i className="fas fa-question-circle text-gray-300 hover:text-accent cursor-help text-sm transition-colors"></i>
    <div className="invisible group-hover:visible absolute z-50 bottom-[140%] left-1/2 -ml-[120px] w-60 bg-primary text-white text-left text-xs p-3 rounded shadow-lg opacity-0 group-hover:opacity-100 transition-opacity pointer-events-none after:content-[''] after:absolute after:top-full after:left-1/2 after:-ml-[5px] after:border-[5px] after:border-solid after:border-t-primary after:border-x-transparent after:border-b-transparent">
      {text}
    </div>
  </div>
);

// Updated InputLabel with Chat Trigger
const InputLabel = ({ label, tooltip, children, onChat }: { label: string, tooltip?: string, children?: React.ReactNode, onChat?: () => void }) => (
  <div className="mb-2">
     <div className="flex justify-between items-end mb-2">
        <label className="font-semibold text-xs uppercase tracking-wide text-gray-500 flex items-center">
        {label} 
        {tooltip && <Tooltip text={tooltip} />}
        </label>
        {onChat && (
            <button 
                onClick={onChat}
                className="flex items-center gap-1.5 px-2 py-1 bg-[#fffbf2] border border-[#f1dca7] rounded-full text-[10px] font-bold text-accent hover:bg-accent hover:text-white transition-colors cursor-pointer shadow-sm"
            >
                <i className="fas fa-sparkles"></i>
                <span className="hidden sm:inline">Hilfe</span>
            </button>
        )}
     </div>
     {children}
  </div>
);

const StdInput = (props: React.InputHTMLAttributes<HTMLInputElement>) => (
  <input 
    {...props}
    className="w-full p-3.5 border border-gray-200 rounded-lg text-base bg-white text-primary transition-all font-sans focus:outline-none focus:border-accent focus:ring-4 focus:ring-accent/10"
  />
);

const StdSelect = (props: React.SelectHTMLAttributes<HTMLSelectElement>) => (
  <select 
    {...props}
    className="w-full p-3.5 border border-gray-200 rounded-lg text-base bg-white text-primary transition-all font-sans appearance-none focus:outline-none focus:border-accent focus:ring-4 focus:ring-accent/10 bg-no-repeat bg-[right_15px_center] bg-[length:16px]"
    style={{ backgroundImage: `url("data:image/svg+xml;charset=UTF-8,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 24' fill='none' stroke='%23333' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3e%3cpolyline points='6 9 12 15 18 9'%3e%3c/polyline%3e%3c/svg%3e")` }}
  />
);


const initialState: ConfigState = {
  checkBauantrag: false,
  checkKfW: false,
  checkSoilReport: false,
  checkSurveying: false,
  checkCrane: false, 
  fundamentSelect: '0',
  fundamentArea: 0,
  areaInput: 0,
  
  wallThicknessExterior: '15',
  wallThicknessInterior: '15',
  
  wallHeight: 245,
  roofSelect: 'none',
  facadeSelect: 'paneele',
  windowMaterial: 'plastic', 
  winPano: 0,
  winStd: 0,
  shutterCount: 0,
  doorOut: 0,
  doorIn: 0,
  glassDoorIn: 0,
  glassDoorFrame: 'plastic',
  drywallSelect: 'none',
  checkCeilingIncluded: true, 
  paintSelect: 'none',
  floorSelect: 'none',
  checkTrittschall: false,
  fireplaceSelect: 'none',
  socketCount: 0,
  lanCount: 0,
  lightStdCount: 0,
  lightSpotCount: 0,
  smokeCount: 0,
  checkFuseBox: false,
  heatSelect: 'none',
  acCount: 0,
  waterType: 'none',
  waterModel: '',
  ventSelect: '0',
  checkPvPrep: false,
  pvSystem: 'none',
  batteryStorage: 'none',
  wcCount: 0,
  sinkCount: 0,
  showerCount: 0,
  tubCount: 0,
  tapCount: 0,
  terraceArea: 0,
  garageCount: 0,
  checkStairOut: false,
  checkStairIn: false,
};

function App() {
  const [config, setConfig] = useState<ConfigState>(initialState);
  const [modalOpen, setModalOpen] = useState(false);
  const [chatOpen, setChatOpen] = useState(false);
  const [chatContext, setChatContext] = useState('');
  
  // Derived state for calculation
  const result: PriceResult = useMemo(() => calculatePrice(config), [config]);

  const handleChange = (field: keyof ConfigState, value: any) => {
    setConfig(prev => ({ ...prev, [field]: value }));
  };

  const getSliderBackground = (val: number, min: number, max: number) => {
    const percentage = ((val - min) / (max - min)) * 100;
    return `linear-gradient(to right, #C5A059 0%, #C5A059 ${percentage}%, #e0e0e0 ${percentage}%, #e0e0e0 100%)`;
  };

  useEffect(() => {
    if (config.waterType === 'none') {
       if (config.waterModel !== '') handleChange('waterModel', '');
    } else if (config.waterType === 'boiler') {
       if (!WATER_BOILER_OPTIONS.includes(Number(config.waterModel))) handleChange('waterModel', WATER_BOILER_OPTIONS[0].toString());
    } else if (config.waterType === 'flow') {
       if (!WATER_FLOW_OPTIONS.includes(config.waterModel)) handleChange('waterModel', WATER_FLOW_OPTIONS[0]);
    }
  }, [config.waterType]);

  const openChat = (context: string) => {
    setChatContext(context);
    setChatOpen(true);
  };

  // --- INTELLIGENT LIVE ADVISOR (Dynamic Tooltips) ---
  const getDynamicTooltip = (key: string, defaultText: string) => {
    switch (key) {
      case 'wallThicknessInterior':
        if (config.wallThicknessExterior === '20') {
          return "Tipp: Bei einer hochgedämmten 20cm Außenhülle empfehlen wir 20cm Innenwände. Diese dienen als thermische Masse und speichern die Wärme effizienter als dünne Wände.";
        }
        if (config.wallThicknessInterior === '12') {
          return "Hinweis: 12cm Wände sind kostengünstig, haben aber kaum Speichermasse. Die Raumtemperatur kann im Sommer schneller steigen und im Winter schneller fallen.";
        }
        return defaultText;

      case 'windowMaterial':
        if (config.wallThicknessExterior === '20' && config.windowMaterial === 'plastic') {
          return "Empfehlung: Bei Ihrer hochwertigen Wanddämmung wären Alu-Fenster technisch die langlebigere Ergänzung.";
        }
        return defaultText;
      
      case 'pvSystem':
        if (config.heatSelect === 'floor_elec') {
          return "Dringende Empfehlung: Bei elektrischer Fußbodenheizung senkt eine PV-Anlage Ihre laufenden Stromkosten massiv!";
        }
        if (config.heatSelect === 'pump') {
          return "Perfekte Kombi: Ihre Wärmepumpe kann den PV-Strom effizient zur Wärmeerzeugung nutzen.";
        }
        return defaultText;

      case 'batteryStorage':
        if (config.pvSystem !== 'none' && config.heatSelect === 'pump') {
          return "Sinnvoll: Speichern Sie den Solarstrom vom Tag, um Ihre Wärmepumpe auch abends effizient zu betreiben.";
        }
        return defaultText;

      case 'fundamentArea':
        if (config.fundamentArea > config.areaInput) {
            return "Hinweis: Ihre Fundamentfläche ist größer als die Wohnfläche. Das ist ungewöhnlich (außer bei großen überdachten Terrassen). Bei 2-stöckigen Häusern ist das Fundament meist ca. 50% der Wohnfläche.";
        }
        if (config.areaInput > 0 && config.fundamentArea < config.areaInput && config.fundamentSelect !== '0') {
          return "Hinweis: Ist Ihr Haus zweistöckig? Dann ist die Fundamentfläche korrekt kleiner als die Wohnfläche.";
        }
        return defaultText;

      default:
        return defaultText;
    }
  };


  // --- HELPER FOR DYNAMIC PRICING DISPLAY ---
  const calcTotal = (unitPrice: number, quantity: number) => {
    if (!quantity) return "0,00 €";
    return formatCurrency(unitPrice * quantity);
  };

  // --- PDF GENERATION LOGIC ---
  const getLabel = (type: string, key: string): string => {
      // Resolve duplicate key 'wood' based on type
      if (type === 'facade' && key === 'wood') return 'Echtholz';
      if (type === 'fireplace' && key === 'wood') return 'Holzkamin (Scheitholz)';

      const map: Record<string, string> = {
          '0': 'Nein / Bauseits', 'none': 'Keine / Bauseits',
          'streifen_ungedaemmt': 'Streifenfundament (ungedämmt)', 'streifen_gedaemmt': 'Streifenfundament (gedämmt)',
          'platte_ungedaemmt': 'Bodenplatte (ungedämmt)', 'platte_gedaemmt': 'Bodenplatte (gedämmt)',
          // Updated Wall Labels
          '12': '12 cm (Slim)', '15': '15 cm (Standard)', '17': '17 cm (Premium)', '20': '20 cm (Passiv / Schallschutz)',
          'flat': 'Flachdach', 'sattel': 'Satteldach', 'pult': 'Pultdach',
          'paneele': 'Standard Paneele', 'wpc': 'WPC Verkleidung', 'putz': 'Putz + Paneele', 'wdvs': 'WDVS verputzt', 
          // 'wood' handled above
          'gips': 'Gipskarton + OSB', 'lehm': 'Baulehm + OSB',
          'plastic': 'Kunststoff (Weiß)', 'alu': 'Aluminium (Anthrazit)',
          'weiss': 'Standard Weiß', 'color': 'Wunschfarbe',
          'vinyl': 'Vinyl', 'laminat': 'Laminat', 'parkett': 'Parkett', 'fliesen': 'Fliesen',
          'pellet': 'Pellet-Kamin', 'electric': 'Elektrischer Kamin',
          'ac': 'Klimaanlage mit Heizfunktion', 'floor_elec': 'Fußbodenhzg. Elektr.', 'floor_water': 'Fußbodenhzg. Wasser', 'pump': 'Wärmepumpe',
          'boiler': 'Boiler', 'flow': 'Durchlauferhitzer',
          'decentral': 'Dezentral', 'central': 'Zentral',
          '5kw': '5 kWp Anlage', '10kw': '10 kWp Anlage',
          '5kwh': '5 kWh Speicher', '10kwh': '10 kWh Speicher'
      };
      return map[key] || key;
  };

  const generatePDF = (name: string, email: string) => {
    // ... (Existing PDF logic unchanged, just keeping it here for context if needed, but in XML update we replace full content)
    setModalOpen(false);
    const doc = new jsPDF();
    const pageWidth = doc.internal.pageSize.width;
    const pageHeight = doc.internal.pageSize.height;
    const colDark = "#1a1a1a";
    const colGold = "#C5A059";
    const colGray = "#f3f3f3";

    doc.setFillColor(colDark);
    doc.rect(0, 0, pageWidth, 40, 'F');
    doc.setTextColor(colGold);
    doc.setFontSize(24);
    doc.setFont("times", "bold");
    doc.text("MODUL MATE GmbH", 20, 20); // Updated to GmbH
    doc.setTextColor(255, 255, 255);
    doc.setFontSize(9);
    doc.setFont("helvetica", "normal");
    doc.text("HAUSKONFIGURATOR", 20, 28); // Removed PREMIUM
    doc.setFontSize(9);
    doc.text("Modul Mate GmbH", pageWidth - 20, 15, { align: "right" });
    doc.text("Cosimastraße 121", pageWidth - 20, 20, { align: "right" });
    doc.text("81925 München", pageWidth - 20, 25, { align: "right" });
    
    let y = 60;
    doc.setTextColor(0, 0, 0);
    doc.setFontSize(8);
    doc.text("Kunde / Bauherr", 20, y - 5);
    doc.setDrawColor(200);
    doc.line(20, y - 3, 80, y - 3);
    doc.setFontSize(11);
    doc.setFont("helvetica", "bold");
    doc.text(name, 20, y + 5);
    doc.setFont("helvetica", "normal");
    doc.text(email, 20, y + 12);
    
    const dateStr = new Date().toLocaleDateString('de-DE');
    doc.setFontSize(10);
    doc.text(`Datum: ${dateStr}`, pageWidth - 20, y, { align: "right" });
    doc.text(`Angebots-Nr.: ${Math.floor(Math.random() * 10000) + 202400}`, pageWidth - 20, y + 6, { align: "right" });
    
    y += 40;
    doc.setFontSize(16);
    doc.setFont("times", "bold");
    doc.setTextColor(colDark);
    doc.text("Ihr persönliches Angebot", 20, y);
    
    doc.setFontSize(10);
    doc.setFont("helvetica", "normal");
    doc.setTextColor(80);
    y += 8;
    doc.text("Basierend auf Ihrer Konfiguration haben wir folgende Kostenaufstellung ermittelt:", 20, y);

    const rows = [
      [
        "1", 
        "Planung & Fundament", 
        `Bauantrag: ${config.checkBauantrag ? 'Ja' : 'Nein'}\nKfW-Beratung: ${config.checkKfW ? 'Ja' : 'Nein'}\nBodengutachten: ${config.checkSoilReport ? 'Ja' : 'Nein'}\nAmtliche Vermessung: ${config.checkSurveying ? 'Ja' : 'Nein'}\nKranstellung: ${config.checkCrane ? 'Ja' : 'Nein (Bauseits)'}\nFundament: ${getLabel('fund', config.fundamentSelect)}${config.fundamentArea > 0 ? ` (${config.fundamentArea} m²)` : ''}`,
        formatCurrency(result.s1)
      ],
      [
        "2", 
        "Haus & Konstruktion", 
        `Wohnfläche: ${config.areaInput} m²\nAußenwand: ${getLabel('wall', config.wallThicknessExterior)}\nInnenwand: ${getLabel('wall', config.wallThicknessInterior)}\nWandhöhe: ${config.wallHeight} cm\nDachform: ${getLabel('roof', config.roofSelect)}\nFassade: ${getLabel('facade', config.facadeSelect)}`,
        formatCurrency(result.s2)
      ],
      [
        "3", 
        "Fenster & Türen", 
        `Standard-Material: ${getLabel('material', config.windowMaterial)}\nPanoramafenster: ${config.winPano} m² (Aluminium)\nStandardfenster: ${config.winStd} m²\nRollläden: ${config.shutterCount} Stk\nHaustür (Alu): ${config.doorOut}\nInnentüren: ${config.doorIn + config.glassDoorIn}`,
        formatCurrency(result.s3)
      ],
      [
        "4", 
        "Innenausbau", 
        `Trockenbau: ${getLabel('dry', config.drywallSelect)}\nMaler: ${getLabel('paint', config.paintSelect)}\nDecke enthalten: ${config.checkCeilingIncluded ? 'Ja' : 'Nein'}\nBoden: ${getLabel('floor', config.floorSelect)}\nKamin: ${getLabel('fireplace', config.fireplaceSelect)}\nElektrik: ${config.socketCount} Steckd., ${config.lightStdCount} Licht\nZählerschrank: ${config.checkFuseBox ? 'Ja' : 'Nein'}`,
        formatCurrency(result.s4)
      ],
      [
        "5", 
        "Technik & Sanitär", 
        `Heizung: ${getLabel('heat', config.heatSelect)}\nWasser: ${getLabel('water', config.waterType)} ${config.waterModel ? `(${config.waterModel})` : ''}\nPV-Vorbereitung: ${config.checkPvPrep ? 'Ja' : 'Nein'}\nPV-Anlage: ${getLabel('pvSystem', config.pvSystem)}\nSpeicher: ${getLabel('batteryStorage', config.batteryStorage)}\nSanitär: ${config.wcCount} WC, ${config.showerCount} Dusche, ${config.sinkCount} WB`,
        formatCurrency(result.s5)
      ],
      [
        "6", 
        "Extras", 
        `Terrassenboden: ${config.terraceArea} m²\nElektrisches Garagentor-Set: ${config.garageCount} Stk\nTreppen: ${config.checkStairOut ? 'Außen ' : ''}${config.checkStairIn ? 'Innen' : ''}`,
        formatCurrency(result.s6)
      ]
    ];

    autoTable(doc, {
      startY: y + 10,
      head: [['Pos', 'Leistungsbereich', 'Details & Spezifikationen', 'Betrag (Netto)']],
      body: rows,
      theme: 'grid',
      headStyles: { fillColor: colDark, textColor: colGold, fontStyle: 'bold', font: 'times', halign: 'left' },
      columnStyles: {
        0: { cellWidth: 15, halign: 'center' },
        1: { cellWidth: 45, fontStyle: 'bold' },
        2: { cellWidth: 'auto', fontSize: 9, cellPadding: 3 },
        3: { cellWidth: 35, halign: 'right', fontStyle: 'bold' }
      },
      styles: { fontSize: 10, cellPadding: 4, lineColor: [230, 230, 230], valign: 'middle', font: 'helvetica' },
      alternateRowStyles: { fillColor: colGray }
    });

    let finalY = (doc as any).lastAutoTable.finalY + 15;
    if (finalY > pageHeight - 60) {
        doc.addPage();
        finalY = 30;
    }
    const boxWidth = 80;
    const boxX = pageWidth - 20 - boxWidth;
    doc.setDrawColor(colGold);
    doc.setLineWidth(0.5);
    doc.line(boxX, finalY, pageWidth - 20, finalY);
    
    finalY += 8;
    doc.setFontSize(11); doc.setTextColor(0);
    doc.text("Gesamtpreis (Netto)", boxX, finalY);
    doc.text(formatCurrency(result.netto), pageWidth - 20, finalY, { align: "right" });
    
    finalY += 7;
    doc.setFontSize(10); doc.setTextColor(100);
    doc.text("zzgl. 19% MwSt", boxX, finalY);
    doc.text(formatCurrency(result.tax), pageWidth - 20, finalY, { align: "right" });
    
    finalY += 8;
    doc.setFillColor(colGray);
    doc.rect(boxX - 2, finalY - 6, boxWidth + 4, 14, 'F');
    doc.setFontSize(14); doc.setFont("times", "bold"); doc.setTextColor(colDark);
    doc.text("Gesamtsumme", boxX, finalY + 1);
    doc.setTextColor(colGold);
    doc.text(formatCurrency(result.brutto), pageWidth - 20, finalY + 1, { align: "right" });
    
    const footerY = pageHeight - 20;
    doc.setDrawColor(200); doc.setLineWidth(0.1);
    doc.line(20, footerY - 5, pageWidth - 20, footerY - 5);
    doc.setFontSize(7); doc.setTextColor(150); doc.setFont("helvetica", "normal");
    const footerText = "Modul Mate GmbH | Cosimastraße 121, 81925 München | GF: Max Mustermann\nHRB: 123456 | Amtsgericht München | USt-IdNr.: DE 123 456 789\nBankverbindung: IBAN DE00 0000 0000 0000 0000 00 | BIC: XXXXXXXX";
    doc.text(footerText, pageWidth / 2, footerY, { align: "center", lineHeightFactor: 1.4 });

    doc.save("ModulMate_Expose_Premium.pdf");
  };

  return (
    <div className="min-h-screen pb-20 lg:pb-0 bg-[#fdfdfd]">
      <LeadModal isOpen={modalOpen} onClose={() => setModalOpen(false)} onSubmit={generatePDF} />
      <ChatWidget isOpen={chatOpen} onClose={() => setChatOpen(false)} context={chatContext} />

      {/* HEADER HERO */}
      <header className="text-center py-20 px-5 bg-primary text-white relative overflow-hidden shadow-2xl">
        <div className="absolute inset-0 opacity-10 pointer-events-none bg-[radial-gradient(#C5A059_1.5px,transparent_1.5px)] bg-[size:40px_40px]"></div>
        <div className="absolute top-0 left-0 w-full h-full bg-gradient-to-b from-transparent to-black/30 pointer-events-none"></div>
        <div className="relative z-10">
            <h1 className="m-0 text-5xl md:text-6xl font-semibold text-accent tracking-wider leading-tight font-serif drop-shadow-lg">Modul Mate GmbH</h1>
            <p className="text-gray-300 text-sm font-light mt-3 font-sans tracking-wide">Cosimastraße 121, 81925 München</p>
            <div className="w-16 h-[2px] bg-accent mx-auto my-6 opacity-80"></div>
            <p className="text-white text-xl font-light tracking-[0.2em] uppercase text-shadow-sm">Hauskonfigurator</p>
        </div>
      </header>

      {/* STICKY NAVIGATION */}
      <Navigation />

      <div className="max-w-[1300px] mx-auto my-12 px-5 grid grid-cols-1 lg:grid-cols-[1fr_380px] gap-12 items-start">
        
        {/* LEFT COLUMN: SECTIONS */}
        <div className="flex flex-col gap-10">

          {/* 1. PLANUNG & FUNDAMENT */}
          <div id="section-1" className="scroll-mt-32">
            <SectionCard number={1} title="Planung & Fundament" price={result.s1}>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <CheckboxCard 
                    checked={config.checkBauantrag}
                    onChange={(v) => handleChange('checkBauantrag', v)}
                    onChat={() => openChat('Bauantrag und Genehmigungen')}
                    title="Bauantrag & Dokumente"
                    subtitle={`Pauschal: ${formatCurrency(PRICES.bauantrag)}`}
                    tooltip="Wir erstellen alle nötigen Unterlagen inkl. Statik & Wärmeschutznachweis für Ihre Baugenehmigung."
                />
                <CheckboxCard 
                    checked={config.checkKfW}
                    onChange={(v) => handleChange('checkKfW', v)}
                    onChat={() => openChat('KfW Förderung und Energieberater')}
                    title="KfW 40 Energieberater"
                    subtitle={`Gesamt: ${calcTotal(PRICES.kfw, config.areaInput)}`}
                    tooltip={`79 €/m² x ${config.areaInput}m². Zwingend nötig für KfW-Förderkredite (297/298).`}
                />
                </div>

                <div className="mt-6 grid grid-cols-1 md:grid-cols-2 gap-6">
                <CheckboxCard 
                    checked={config.checkSoilReport}
                    onChange={(v) => handleChange('checkSoilReport', v)}
                    onChat={() => openChat('Bodengutachten')}
                    title="Bodengutachten"
                    subtitle={formatCurrency(PRICES.soilReport)}
                    compact
                    tooltip="Analyse der Bodenbeschaffenheit zur Festlegung der notwendigen Fundamentart und Abdichtung."
                    />
                <CheckboxCard 
                    checked={config.checkSurveying}
                    onChange={(v) => handleChange('checkSurveying', v)}
                    onChat={() => openChat('Amtliche Vermessung')}
                    title="Amtliche Vermessung"
                    subtitle={formatCurrency(PRICES.surveying)}
                    compact
                    tooltip="Einmessung des Gebäudes durch ein Vermessungsbüro (Grobabsteckung & Schnurgerüst) für das Bauamt."
                    />
                </div>

                <div className="mt-6">
                    <CheckboxCard 
                        checked={config.checkCrane}
                        onChange={(v) => handleChange('checkCrane', v)}
                        onChat={() => openChat('Kranstellung')}
                        title="Kranstellung & Straßensperrung"
                        subtitle={`${formatCurrency(PRICES.crane)} (Pauschal)`}
                        compact
                        tooltip="Notwendig, wenn der LKW nicht direkt an die Bodenplatte fahren kann (z.B. bei Hinterliegergrundstücken) oder eine Straßensperrung nötig ist."
                    />
                </div>
                
                <InputLabel label="Fundament" onChat={() => openChat('Fundamentarten')} tooltip={getDynamicTooltip('fundamentArea', "Die Basis Ihres Hauses. Preise werden basierend auf der angegebenen Fundamentfläche berechnet.")}>
                <div className="grid grid-cols-1 md:grid-cols-[2fr_1fr] gap-5 mt-4">
                    <div>
                    <small className="block text-accent text-xs mt-1 mb-1 font-bold uppercase">Art auswählen</small>
                    <StdSelect value={config.fundamentSelect} onChange={(e) => handleChange('fundamentSelect', e.target.value)}>
                        <option value="0">Kein Fundament (0,00 €)</option>
                        <option value="streifen_ungedaemmt">Streifenfundament ungedämmt ({calcTotal(PRICES.fund.streifen_ungedaemmt, config.fundamentArea)})</option>
                        <option value="streifen_gedaemmt">Streifenfundament gedämmt ({calcTotal(PRICES.fund.streifen_gedaemmt, config.fundamentArea)})</option>
                        <option value="platte_ungedaemmt">Betonbodenplatte ungedämmt ({calcTotal(PRICES.fund.platte_ungedaemmt, config.fundamentArea)})</option>
                        <option value="platte_gedaemmt">Betonbodenplatte gedämmt ({calcTotal(PRICES.fund.platte_gedaemmt, config.fundamentArea)})</option>
                    </StdSelect>
                    </div>
                    <div>
                    <small className="flex items-center gap-1 text-accent text-xs mt-1 mb-1 font-bold uppercase">
                        Fläche (m²) <Tooltip text={getDynamicTooltip('fundamentArea', "Geben Sie hier die tatsächliche Bodenplatten-Größe an. Diese kann von der Wohnfläche abweichen (z.B. bei 2-stöckiger Bauweise).")} />
                    </small>
                    <StdInput type="number" placeholder="0" value={config.fundamentArea || ''} onChange={(e) => handleChange('fundamentArea', parseFloat(e.target.value))} />
                    </div>
                </div>
                </InputLabel>
            </SectionCard>
          </div>

          {/* 2. HAUS & KONSTRUKTION */}
          <div id="section-2" className="scroll-mt-32">
            <SectionCard number={2} title="Haus & Konstruktion" price={result.s2}>
                {/* Energy Dashboard */}
                <div className="bg-gray-50 border border-gray-200 rounded-xl p-6 mb-2 shadow-inner">
                    <div className="flex justify-between items-center mb-2">
                    <span className="text-sm font-bold tracking-wide flex items-center text-primary">
                        <i className="fas fa-leaf text-success mr-2"></i> ENERGIE-EFFIZIENZ
                        <Tooltip text="Dieser Score zeigt die energetische Qualität Ihres Hauses (0-100). Er berücksichtigt stark die Kombination aus Außenwanddämmung, Heiztechnik, Lüftung und die thermische Masse der Innenwände." />
                    </span>
                    <button onClick={() => openChat('Energieeffizienz und U-Werte')} className="text-xs bg-white border border-gray-200 px-2 py-1 rounded-md text-accent font-bold hover:bg-accent hover:text-white transition-colors">
                        <i className="fas fa-sparkles mr-1"></i> Energie-Berater
                    </button>
                    </div>
                    <div className="flex justify-between items-center mb-1">
                         <span className="font-serif font-bold text-primary text-lg">U-Wert: {result.uValue}</span>
                    </div>
                    <div className="h-3 bg-gray-200 rounded-full overflow-hidden my-3 shadow-inner">
                    <div 
                        className="h-full bg-gradient-to-r from-red-500 via-amber-500 to-green-500 transition-all duration-1000 ease-out"
                        style={{ width: `${result.energyScore}%` }}
                    ></div>
                    </div>
                    <div className="flex justify-between items-center">
                    <span className="text-xs font-semibold text-gray-500 uppercase tracking-widest">{result.energyStandard}</span>
                    <small className="text-[10px] text-gray-400 italic">Planungs-Richtwert</small>
                    </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6">
                   <div>
                        <InputLabel label="Wohnfläche (m²)" onChat={() => openChat('Wohnfläche')} tooltip="Die gesamte beheizte Wohnfläche über alle Etagen. Basis für die meisten Kalkulationen.">
                        <StdInput type="number" placeholder="0" value={config.areaInput || ''} onChange={(e) => handleChange('areaInput', parseFloat(e.target.value))} />
                        </InputLabel>
                        <div className="mt-2 p-3 bg-blue-50 border border-blue-100 rounded-lg text-blue-800 text-xs flex items-start gap-2">
                            <i className="fas fa-calculator mt-0.5"></i>
                            <span>Wandfläche: ca. {Math.round(result.wallArea)} m² (Kalkuliert für Fassade)</span>
                        </div>
                   </div>
                   
                   <div className="space-y-4">
                        <InputLabel label="Wand-Kalkulator" onChat={() => openChat('Wandaufbau und Dämmung')} tooltip="Genaue Kostenberechnung basierend auf Ihrer Wohnfläche.">
                            <div className="bg-gray-50 p-4 rounded-lg border border-gray-200">
                                
                                {/* Außenwand */}
                                <div className="mb-4 pb-4 border-b border-gray-200">
                                    <small className="block text-accent text-[10px] mb-2 font-bold uppercase">Außenwand (Gesamtpreis)</small>
                                    <div className="space-y-2">
                                        {[
                                            { val: '15', label: '15 cm', price: PRICES.wallExterior['15'] },
                                            { val: '17', label: '17 cm', price: PRICES.wallExterior['17'] },
                                            { val: '20', label: '20 cm', price: PRICES.wallExterior['20'] }
                                        ].map((opt) => (
                                            <label key={opt.val} className={`flex justify-between items-center p-2 rounded cursor-pointer border ${config.wallThicknessExterior === opt.val ? 'bg-white border-accent shadow-sm' : 'border-transparent hover:bg-gray-100'}`}>
                                                <div className="flex items-center gap-2">
                                                    <input type="radio" name="wallExt" checked={config.wallThicknessExterior === opt.val} onChange={() => handleChange('wallThicknessExterior', opt.val)} className="accent-accent" />
                                                    <span className="text-sm">{opt.label}</span>
                                                </div>
                                                <span className="text-sm font-bold">{calcTotal(opt.price, config.areaInput)}</span>
                                            </label>
                                        ))}
                                    </div>
                                </div>

                                {/* Innenwand */}
                                <div>
                                    <small className="block text-accent text-[10px] mb-2 font-bold uppercase flex items-center">
                                        Innenwand (Auf-/Abschlag) 
                                        <Tooltip text={getDynamicTooltip('wallThicknessInterior', 'Wählen Sie die Stärke Ihrer Innenwände.')} />
                                    </small>
                                    <div className="space-y-2">
                                        {[
                                            { val: '12', label: '12 cm (Slim)', price: PRICES.innerWallSurcharge['12'] },
                                            { val: '15', label: '15 cm (Standard)', price: PRICES.innerWallSurcharge['15'] },
                                            { val: '20', label: '20 cm (Schallschutz)', price: PRICES.innerWallSurcharge['20'] }
                                        ].map((opt) => {
                                            const total = opt.price * config.areaInput;
                                            const isDiscount = total < 0;
                                            const isZero = total === 0;
                                            return (
                                                <label key={opt.val} className={`flex justify-between items-center p-2 rounded cursor-pointer border ${config.wallThicknessInterior === opt.val ? 'bg-white border-accent shadow-sm' : 'border-transparent hover:bg-gray-100'}`}>
                                                    <div className="flex items-center gap-2">
                                                        <input type="radio" name="wallInt" checked={config.wallThicknessInterior === opt.val} onChange={() => handleChange('wallThicknessInterior', opt.val)} className="accent-accent" />
                                                        <span className="text-sm">{opt.label}</span>
                                                    </div>
                                                    <span className={`text-sm font-bold ${isDiscount ? 'text-success' : isZero ? 'text-gray-400' : 'text-primary'}`}>
                                                        {isZero ? 'Inklusive' : formatCurrency(total)}
                                                    </span>
                                                </label>
                                            )
                                        })}
                                    </div>
                                </div>

                            </div>
                        </InputLabel>
                   </div>
                </div>

                <div className="mt-8">
                <InputLabel label="Innenwandhöhe" onChat={() => openChat('Raumhöhe')} tooltip="Standard ist 245cm. Höhere Decken sorgen für ein großzügigeres Raumgefühl, erhöhen aber das zu beheizende Volumen.">
                    <span className="text-accent font-bold ml-1 text-xl font-serif">{config.wallHeight}</span> cm
                    <small className="inline font-normal ml-2 text-gray-400 text-xs">
                        {config.wallHeight > 245 
                            ? `(Aufpreis: ${formatCurrency((config.wallHeight - 245) * PRICES.height * config.areaInput)})` 
                            : '(Standard)'}
                    </small>
                </InputLabel>
                <div className="py-2">
                    <input 
                    type="range" 
                    min="245" max="330" 
                    value={config.wallHeight} 
                    onChange={(e) => handleChange('wallHeight', parseInt(e.target.value))}
                    style={{ background: getSliderBackground(config.wallHeight, 245, 330) }}
                    className="w-full h-2 rounded-lg appearance-none cursor-pointer"
                    />
                </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6">
                    <InputLabel label="Dachform" onChat={() => openChat('Dachformen')} tooltip="Sattel- und Pultdächer bieten zusätzlichen Stauraum im Dachboden. Flachdach ist modern und kosteneffizient.">
                    <StdSelect value={config.roofSelect} onChange={(e) => handleChange('roofSelect', e.target.value)}>
                        <option value="none">Kein Dach (0,00 €)</option>
                        <option value="flat">Flachdach ({calcTotal(PRICES.roof.flat, config.areaInput)})</option>
                        <option value="sattel">Satteldach ({calcTotal(PRICES.roof.sattel, config.areaInput)})</option>
                        <option value="pult">Pultdach ({calcTotal(PRICES.roof.pult, config.areaInput)})</option>
                    </StdSelect>
                    </InputLabel>
                    <InputLabel label="Fassade" onChat={() => openChat('Fassadenarten')} tooltip="Preise basieren auf der berechneten Fassadenfläche. WPC und Echtholz bieten eine hochwertige Optik.">
                        {/* Fassade uses calculated wallArea, not areaInput directly */}
                        <StdSelect value={config.facadeSelect} onChange={(e) => handleChange('facadeSelect', e.target.value)}>
                            <option value="paneele">Standard Paneele (Inklusive)</option>
                            <option value="wpc">WPC Verkleidung (+{calcTotal(PRICES.facade.wpc, result.wallArea)})</option>
                            <option value="putz">Putz + Paneele (+{calcTotal(PRICES.facade.putz, result.wallArea)})</option>
                            <option value="wdvs">WDVS verputzt (+{calcTotal(PRICES.facade.wdvs, result.wallArea)})</option>
                            <option value="wood">Echtholz (+{calcTotal(PRICES.facade.wood, result.wallArea)})</option>
                            <option value="none">Keine / Bauseits (Eigenleistung)</option>
                        </StdSelect>
                    </InputLabel>
                </div>
            </SectionCard>
          </div>

          {/* 3. FENSTER */}
          <div id="section-3" className="scroll-mt-32">
            <SectionCard number={3} title="Fenster & Türen" price={result.s3}>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <InputLabel label="Panoramafenster (m²)" onChat={() => openChat('Fensterflächen')} tooltip="Bodentiefe Fenster. Ausführung immer in hochwertigem Aluminium (Anthrazit) für Stabilität.">
                        <small className="text-gray-400 text-xs mb-1 block">
                            Material: Aluminium (Anthrazit)
                        </small>
                        <small className="text-accent text-xs mb-1 block">
                            Gesamt: {formatCurrency(config.winPano * PRICES.win.pano)}
                        </small>
                        <StdInput type="number" placeholder="0" value={config.winPano || ''} onChange={(e) => handleChange('winPano', parseFloat(e.target.value))} />
                    </InputLabel>

                    {/* Standard Window with Integrated Material Selection */}
                    <div>
                        <InputLabel label="Standardfenster 3-fach (m²)" onChat={() => openChat('Fenstermaterial')} tooltip={getDynamicTooltip('windowMaterial', "Klassische Fenstergrößen für Schlafzimmer und Nebenräume.")}>
                            <div className="flex bg-gray-50 border border-gray-200 p-1 rounded-lg mb-2">
                                <button 
                                    onClick={() => handleChange('windowMaterial', 'plastic')}
                                    className={`flex-1 py-1.5 text-xs font-bold rounded-md transition-all ${config.windowMaterial === 'plastic' ? 'bg-white text-primary shadow-sm ring-1 ring-black/5' : 'text-gray-500 hover:text-gray-700'}`}
                                >
                                    <i className="fas fa-square mr-1"></i> Kunststoff
                                </button>
                                <button 
                                    onClick={() => handleChange('windowMaterial', 'alu')}
                                    className={`flex-1 py-1.5 text-xs font-bold rounded-md transition-all ${config.windowMaterial === 'alu' ? 'bg-white text-primary shadow-sm ring-1 ring-black/5' : 'text-gray-500 hover:text-gray-700'}`}
                                >
                                    <i className="fas fa-layer-group mr-1"></i> Alu (+9%)
                                </button>
                            </div>
                            <small className="text-accent text-xs mb-1 block">
                                Gesamt: {formatCurrency(config.winStd * PRICES.win.std * PRICES.winMaterialFactor[config.windowMaterial])}
                            </small>
                            <StdInput type="number" placeholder="0" value={config.winStd || ''} onChange={(e) => handleChange('winStd', parseFloat(e.target.value))} />
                        </InputLabel>
                    </div>
                </div>
                
                <InputLabel label="Elektr. Rollläden (Stk)" onChat={() => openChat('Rollläden und Verschattung')} tooltip="Elektrisch betrieben für Komfort und Sicherheit. Empfohlen für Schlafbereiche zur Verdunklung.">
                    <small className="text-accent text-xs mb-1 block">
                        Gesamt: {calcTotal(PRICES.win.shutter, config.shutterCount)}
                    </small>
                    <StdInput type="number" placeholder="0" value={config.shutterCount || ''} onChange={(e) => handleChange('shutterCount', parseFloat(e.target.value))} />
                </InputLabel>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-5 mt-4">
                    <InputLabel label="Haustür Alu" onChat={() => openChat('Haustüren')} tooltip="Hochwertige Aluminium-Haustür mit Mehrfachverriegelung für Sicherheit und guten Wärmeschutz.">
                    <small className="text-accent text-xs mb-1 block">
                        Gesamt: {calcTotal(PRICES.door.out, config.doorOut)}
                    </small>
                    <StdInput type="number" placeholder="0" value={config.doorOut || ''} onChange={(e) => handleChange('doorOut', parseFloat(e.target.value))} />
                    </InputLabel>
                    <InputLabel label="Innentüren" onChat={() => openChat('Innentüren')} tooltip="Standard CPL-Innentüren (Weiß) inklusive Zarge und Drückergarnitur. Robust und pflegeleicht.">
                    <small className="text-accent text-xs mb-1 block">
                        Gesamt: {calcTotal(PRICES.door.in, config.doorIn)}
                    </small>
                    <StdInput type="number" placeholder="0" value={config.doorIn || ''} onChange={(e) => handleChange('doorIn', parseFloat(e.target.value))} />
                    </InputLabel>
                    <div>
                    <InputLabel label="Glastüren" onChat={() => openChat('Glastüren')} tooltip="Lassen Licht in den Flur oder Verbindungsgänge. Wahlweise mit Kunststoff- oder Alurahmen.">
                        <small className="text-accent text-xs mb-1 block">
                            Gesamt: {calcTotal(config.glassDoorFrame === 'plastic' ? PRICES.door.glassKst : PRICES.door.glassAlu, config.glassDoorIn)}
                        </small>
                        <div className="flex gap-2">
                            <input type="number" className="w-[80px] p-3.5 border border-gray-200 rounded-lg bg-white focus:border-accent focus:outline-none" placeholder="0" value={config.glassDoorIn || ''} onChange={(e) => handleChange('glassDoorIn', parseFloat(e.target.value))} />
                            <select className="flex-1 p-3.5 border border-gray-200 rounded-lg bg-white focus:border-accent focus:outline-none" value={config.glassDoorFrame} onChange={(e) => handleChange('glassDoorFrame', e.target.value)}>
                            <option value="plastic">Kst.</option>
                            <option value="alu">Alu</option>
                            </select>
                        </div>
                    </InputLabel>
                    </div>
                </div>
            </SectionCard>
          </div>

          {/* 4. INNENAUSBAU */}
          <div id="section-4" className="scroll-mt-32">
            <SectionCard number={4} title="Innenausbau" price={result.s4}>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6 items-start">
                    <div>
                        {/* Interior calc Area for Drywall/Paint approx 2.5x base area + ceiling */}
                        <InputLabel label="Trockenbau" onChat={() => openChat('Trockenbau Materialien')} tooltip="Beplankung der Wände. Lehmbauplatten regulieren die Feuchtigkeit besser und sorgen für ein hervorragendes Raumklima.">
                        <StdSelect value={config.drywallSelect} onChange={(e) => handleChange('drywallSelect', e.target.value)}>
                            <option value="none">Kein Trockenbau (0,00 €)</option>
                            <option value="gips">Gipskarton + OSB ({formatCurrency((config.areaInput * (config.checkCeilingIncluded ? 3.5 : 2.5)) * PRICES.dry.gips)})</option>
                            <option value="lehm">OSB + Baulehmplatten ({formatCurrency((config.areaInput * (config.checkCeilingIncluded ? 3.5 : 2.5)) * PRICES.dry.lehm)})</option>
                        </StdSelect>
                        </InputLabel>
                    </div>
                    <div>
                        <InputLabel label="Malerarbeiten" onChat={() => openChat('Malerarbeiten')} tooltip="Verspachteln (Q3) und Streichen. Standard ist Weiß. Bei 'Eigenleistung' werden die Wände nur Q2 vorbereitet übergeben.">
                        <StdSelect value={config.paintSelect} onChange={(e) => handleChange('paintSelect', e.target.value)}>
                            <option value="none">Keine (0,00 €)</option>
                            <option value="weiss">Schwarz/Weiß ({formatCurrency((config.areaInput * (config.checkCeilingIncluded ? 3.5 : 2.5)) * PRICES.paint.std)})</option>
                            <option value="color">Wunschfarbe ({formatCurrency((config.areaInput * (config.checkCeilingIncluded ? 3.5 : 2.5)) * PRICES.paint.std * PRICES.paint.factor)})</option>
                        </StdSelect>
                        </InputLabel>
                    </div>
                </div>

                <div className="mt-4 mb-6">
                    <CheckboxCard 
                        checked={config.checkCeilingIncluded}
                        onChange={(v) => handleChange('checkCeilingIncluded', v)}
                        onChat={() => openChat('Deckenverkleidung')}
                        title="Deckenflächen einbeziehen"
                        subtitle="Trockenbau & Malerarbeiten auch für die Decke"
                        compact
                        tooltip="Empfohlen. Wenn abgewählt, werden nur die Wände bearbeitet, die Decke bleibt im Rohzustand."
                    />
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6">
                    <InputLabel label="Bodenbelag" onChat={() => openChat('Bodenbeläge')} tooltip="Vinyl ist robust und wasserfest. Parkett ist warm und natürlich. Laminat ist die günstige Alternative. Fliesen für Nassbereiche.">
                    <StdSelect value={config.floorSelect} onChange={(e) => handleChange('floorSelect', e.target.value)}>
                        <option value="none">Kein Bodenbelag (0,00 €)</option>
                        <option value="vinyl">Vinyl ({calcTotal(PRICES.floor.vinyl, config.areaInput)})</option>
                        <option value="laminat">Laminat ({calcTotal(PRICES.floor.laminat, config.areaInput)})</option>
                        <option value="parkett">Parkett ({calcTotal(PRICES.floor.parkett, config.areaInput)})</option>
                        <option value="fliesen">Fliesen ({calcTotal(PRICES.floor.fliesen, config.areaInput)})</option>
                    </StdSelect>
                    </InputLabel>
                    {config.floorSelect === 'laminat' && (
                    <div>
                        <InputLabel label="Extras Boden" />
                        <CheckboxCard 
                        checked={config.checkTrittschall}
                        onChange={(v) => handleChange('checkTrittschall', v)}
                        onChat={() => openChat('Trittschalldämmung')}
                        title="Trittschalldämmung"
                        subtitle={`+${calcTotal(PRICES.trittschall, config.areaInput)}`}
                        compact
                        tooltip="Reduziert den Gehschall deutlich und gleicht kleine Unebenheiten aus. Dringend empfohlen für Laminat."
                        />
                    </div>
                    )}
                </div>

                <div className="mt-6 mb-8 bg-gray-50 border border-gray-200 p-4 rounded-lg">
                    <InputLabel label="Kamin & Ambiente" onChat={() => openChat('Kamin')} tooltip="Ein Kamin sorgt für wohlige Wärme und eine exklusive Atmosphäre. Preise erhalten Sie im individuellen Angebot.">
                        <StdSelect value={config.fireplaceSelect} onChange={(e) => handleChange('fireplaceSelect', e.target.value)}>
                            <option value="none">Kein Kamin</option>
                            <option value="wood">Holzkamin (Scheitholz)</option>
                            <option value="pellet">Pellet-Kamin</option>
                            <option value="electric">Elektrischer Kamin</option>
                        </StdSelect>
                    </InputLabel>
                    {config.fireplaceSelect !== 'none' && (
                        <div className="mt-2 text-xs text-orange-600 font-medium flex items-center gap-2">
                            <i className="fas fa-info-circle"></i>
                            Preis auf Anfrage (Individuelles Angebot)
                        </div>
                    )}
                </div>

                <h4 className="mt-8 border-b border-gray-100 pb-2 text-lg text-primary font-serif">Elektrik & Sicherheit</h4>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-5 mt-4">
                <InputLabel label="Steckdosen" onChat={() => openChat('Elektroplanung Steckdosen')} tooltip="Planen Sie ca. 3-4 Steckdosen pro Raum ein. In der Küche meist mehr.">
                    <small className="text-accent text-xs mb-1 block">Gesamt: {calcTotal(PRICES.elec.socket, config.socketCount)}</small>
                    <StdInput type="number" placeholder="0" value={config.socketCount || ''} onChange={(e) => handleChange('socketCount', parseFloat(e.target.value))} />
                </InputLabel>
                <InputLabel label="LAN-Dosen" onChat={() => openChat('Netzwerkplanung')} tooltip="Für stabiles Internet in Homeoffice und Wohnzimmer (CAT 7 Verkabelung).">
                    <small className="text-accent text-xs mb-1 block">Gesamt: {calcTotal(PRICES.elec.lan, config.lanCount)}</small>
                    <StdInput type="number" placeholder="0" value={config.lanCount || ''} onChange={(e) => handleChange('lanCount', parseFloat(e.target.value))} />
                </InputLabel>
                <InputLabel label="Licht (Standard)" onChat={() => openChat('Beleuchtung Standard')} tooltip="Klassischer Deckenauslass (Kabel) in der Raummitte für Hängelampen.">
                    <small className="text-accent text-xs mb-1 block">Gesamt: {calcTotal(PRICES.elec.lightStd, config.lightStdCount)}</small>
                    <StdInput type="number" placeholder="0" value={config.lightStdCount || ''} onChange={(e) => handleChange('lightStdCount', parseFloat(e.target.value))} />
                </InputLabel>
                <InputLabel label="Spots" onChat={() => openChat('Beleuchtung Spots')} tooltip="Moderne, in die Decke eingelassene LED-Spots für gleichmäßige Ausleuchtung.">
                    <small className="text-accent text-xs mb-1 block">Gesamt: {calcTotal(PRICES.elec.lightSpot, config.lightSpotCount)}</small>
                    <StdInput type="number" placeholder="0" value={config.lightSpotCount || ''} onChange={(e) => handleChange('lightSpotCount', parseFloat(e.target.value))} />
                </InputLabel>
                <InputLabel label="Rauchmelder" onChat={() => openChat('Rauchmelder')} tooltip="Gesetzlich vorgeschrieben in Schlafzimmern, Kinderzimmern und Fluren.">
                    <small className="text-accent text-xs mb-1 block">Gesamt: {calcTotal(PRICES.elec.smoke, config.smokeCount)}</small>
                    <StdInput type="number" placeholder="0" value={config.smokeCount || ''} onChange={(e) => handleChange('smokeCount', parseFloat(e.target.value))} />
                </InputLabel>
                
                <div>
                    <InputLabel label="Zählerschrank VDE" onChat={() => openChat('Zählerschrank und VDE')} tooltip="Der zentrale Sicherungskasten mit FI-Schutzschalter, fertig vorverdrahtet nach deutscher VDE-Norm."><small className="text-accent text-xs mb-1 block">390 €</small></InputLabel>
                    <CheckboxCard 
                        checked={config.checkFuseBox}
                        onChange={(v) => handleChange('checkFuseBox', v)}
                        title="Installation"
                        subtitle="Ja"
                        compact
                        tooltip="Ohne Zählerschrank kann das Haus nicht ans Stromnetz angeschlossen werden."
                    />
                </div>
                </div>
            </SectionCard>
          </div>

          {/* 5. TECHNIK */}
          <div id="section-5" className="scroll-mt-32">
            <SectionCard number={5} title="Technik & Sanitär" price={result.s5}>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <InputLabel label="Heizsystem" onChat={() => openChat('Heizsysteme')} tooltip="Wärmepumpe ist am effizientesten und zukunftssicher. Fußbodenheizung spart Platz und bietet hohen Komfort. Split-Klimageräte heizen und kühlen schnell.">
                    <StdSelect value={config.heatSelect} onChange={(e) => handleChange('heatSelect', e.target.value)}>
                        <option value="none">Keine / Bauseits (0,00 €)</option>
                        <option value="ac">Klimaanlage mit Heizfunktion ({calcTotal(PRICES.ac, config.acCount)})</option>
                        <option value="floor_elec">Fußbodenhzg. Elektr. ({calcTotal(PRICES.heatFloor.el, config.areaInput)})</option>
                        <option value="floor_water">Fußbodenhzg. Wasser ({calcTotal(PRICES.heatFloor.wa, config.areaInput)})</option>
                        <option value="pump">Wärmepumpe Komplett ({calcTotal(PRICES.heatPump, config.areaInput * 0.05)})</option>
                    </StdSelect>
                    </InputLabel>
                    {config.heatSelect === 'ac' && (
                    <InputLabel label="Anzahl Split-Geräte" tooltip="Anzahl der Innengeräte, die im Haus verteilt installiert werden.">
                        <StdInput type="number" placeholder="0" value={config.acCount || ''} onChange={(e) => handleChange('acCount', parseFloat(e.target.value))} />
                    </InputLabel>
                    )}
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6">
                    <InputLabel label="Warmwasser Art" onChat={() => openChat('Warmwasseraufbereitung')} tooltip="Boiler speichert warmes Wasser (Platzbedarf). Durchlauferhitzer erwärmt es bei Bedarf (hoher Stromanschluss nötig).">
                    <StdSelect value={config.waterType} onChange={(e) => handleChange('waterType', e.target.value)}>
                        <option value="none">Kein System (Bauseits/Eigenl.)</option>
                        <option value="boiler">Boiler (Speicher)</option>
                        <option value="flow">Durchlauferhitzer</option>
                    </StdSelect>
                    </InputLabel>
                    <InputLabel label="Modell / Leistung" onChat={() => openChat('Warmwasser Dimensionierung')} tooltip="Wählen Sie die Größe passend zur Personenanzahl. (ca. 40-50L Speicher pro Person empfohlen).">
                    <StdSelect value={config.waterModel} onChange={(e) => handleChange('waterModel', e.target.value)} disabled={config.waterType === 'none'}>
                        {config.waterType === 'none' && <option value="">-</option>}
                        {config.waterType === 'boiler' && WATER_BOILER_OPTIONS.map(s => <option key={s} value={s}>{s} Liter ({formatCurrency(PRICES.water.boiler[String(s) as keyof typeof PRICES.water.boiler])})</option>)}
                        {config.waterType === 'flow' && WATER_FLOW_OPTIONS.map(k => <option key={k} value={k}>{k} kW ({formatCurrency(PRICES.water.flow[k as keyof typeof PRICES.water.flow])})</option>)}
                    </StdSelect>
                    </InputLabel>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6 items-start">
                <div>
                    <InputLabel label="Lüftung" onChat={() => openChat('Lüftungsanlagen')} tooltip="Eine Lüftungsanlage verhindert Schimmelbildung und spart Heizkosten durch Wärmerückgewinnung.">
                    <StdSelect value={config.ventSelect} onChange={(e) => handleChange('ventSelect', e.target.value)}>
                        <option value="0">Keine</option>
                        <option value="decentral">Dezentrale Lüftung (Anfrage/Stück)</option>
                        <option value="central">Zentrale Anlage (Anfrage/Pauschal)</option>
                    </StdSelect>
                    </InputLabel>
                    {config.ventSelect !== '0' && <small className="text-orange-500 text-xs mt-1 block">Preis auf Anfrage (nicht im Total enthalten)</small>}
                </div>
                
                <div>
                    <InputLabel label="PV-Vorbereitung" tooltip="Leerrohre und Kabelwege vom Dach zum Technikraum, um eine Solaranlage später einfach nachrüsten zu können."><small className="text-accent text-xs mb-1 block">450 €</small></InputLabel>
                    <CheckboxCard 
                        checked={config.checkPvPrep}
                        onChange={(v) => handleChange('checkPvPrep', v)}
                        onChat={() => openChat('PV-Vorbereitung')}
                        title="Installation"
                        subtitle="Leerrohre & Kabel"
                        compact
                        tooltip="Empfohlen, wenn Sie später Photovoltaik nachrüsten möchten, ohne Wände aufzureißen."
                    />
                </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6">
                    <InputLabel label="PV-Anlage (inkl. Module)" onChat={() => openChat('Photovoltaikanlage')} tooltip={getDynamicTooltip('pvSystem', "Produzieren Sie Ihren eigenen Strom. Eine 10kWp Anlage deckt meist den Jahresbedarf einer 4-köpfigen Familie.")}>
                    <StdSelect value={config.pvSystem} onChange={(e) => handleChange('pvSystem', e.target.value)}>
                        <option value="none">Keine</option>
                        <option value="5kw">5 kWp Anlage ({formatCurrency(PRICES.pvSystem['5kw'])})</option>
                        <option value="10kw">10 kWp Anlage ({formatCurrency(PRICES.pvSystem['10kw'])})</option>
                    </StdSelect>
                    </InputLabel>
                    <InputLabel label="Batteriespeicher" onChat={() => openChat('Stromspeicher')} tooltip={getDynamicTooltip('batteryStorage', "Speichert den tagsüber produzierten Solarstrom für die Nutzung am Abend und in der Nacht. Erhöht die Autarkie.")}>
                    <StdSelect value={config.batteryStorage} onChange={(e) => handleChange('batteryStorage', e.target.value)}>
                        <option value="none">Keine</option>
                        <option value="5kwh">5 kWh Speicher ({formatCurrency(PRICES.batteryStorage['5kwh'])})</option>
                        <option value="10kwh">10 kWh Speicher ({formatCurrency(PRICES.batteryStorage['10kwh'])})</option>
                    </StdSelect>
                    </InputLabel>
                </div>

                <h4 className="mt-8 border-b border-gray-100 pb-2 text-lg text-primary font-serif">Sanitärobjekte</h4>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-5 mt-4">
                    <InputLabel label="WC" onChat={() => openChat('Sanitär WC')} tooltip="Wandhängendes Tiefspül-WC, spülrandlos für bessere Hygiene, inkl. Sitz mit Absenkautomatik.">
                        <small className="text-accent text-xs mb-1 block">Gesamt: {calcTotal(PRICES.sanitary.wc, config.wcCount)}</small>
                        <StdInput type="number" placeholder="0" value={config.wcCount || ''} onChange={(e) => handleChange('wcCount', parseFloat(e.target.value))} />
                    </InputLabel>
                    <InputLabel label="Waschbecken" onChat={() => openChat('Sanitär Waschbecken')} tooltip="Standard Keramik-Waschbecken (ca. 60cm) inklusive Einhebelmischer-Armatur.">
                        <small className="text-accent text-xs mb-1 block">Gesamt: {calcTotal(PRICES.sanitary.sink, config.sinkCount)}</small>
                        <StdInput type="number" placeholder="0" value={config.sinkCount || ''} onChange={(e) => handleChange('sinkCount', parseFloat(e.target.value))} />
                    </InputLabel>
                    <InputLabel label="Dusche" onChat={() => openChat('Sanitär Dusche')} tooltip="Duschkabine (80x80cm oder 90x90cm) mit Glasabtrennung und Armatur.">
                        <small className="text-accent text-xs mb-1 block">Gesamt: {calcTotal(PRICES.sanitary.shower, config.showerCount)}</small>
                        <StdInput type="number" placeholder="0" value={config.showerCount || ''} onChange={(e) => handleChange('showerCount', parseFloat(e.target.value))} />
                    </InputLabel>
                    <InputLabel label="Badewanne" onChat={() => openChat('Sanitär Badewanne')} tooltip="Klassische Acryl-Badewanne (170x75cm) inklusive Wannenfüllarmatur und Ablaufgarnitur.">
                        <small className="text-accent text-xs mb-1 block">Gesamt: {calcTotal(PRICES.sanitary.tub, config.tubCount)}</small>
                        <StdInput type="number" placeholder="0" value={config.tubCount || ''} onChange={(e) => handleChange('tubCount', parseFloat(e.target.value))} />
                    </InputLabel>
                    <InputLabel label="Außenhahn" onChat={() => openChat('Außenwasserhahn')} tooltip="Frostsicherer Außenwasserhahn für die Gartenbewässerung.">
                        <small className="text-accent text-xs mb-1 block">Gesamt: {calcTotal(PRICES.sanitary.tap, config.tapCount)}</small>
                        <StdInput type="number" placeholder="0" value={config.tapCount || ''} onChange={(e) => handleChange('tapCount', parseFloat(e.target.value))} />
                    </InputLabel>
                </div>
            </SectionCard>
          </div>

          {/* 6. EXTRAS */}
          <div id="section-6" className="scroll-mt-32">
            <SectionCard number={6} title="Extras" price={result.s6}>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <InputLabel label="Terrassenboden (m²)" onChat={() => openChat('Terrasse')} tooltip="Inklusive Unterkonstruktion und Belag (Holz oder WPC). Erweiterung der Wohnfläche in den Außenbereich.">
                    <small className="text-accent text-xs mb-1 block">
                        Gesamt: {calcTotal(PRICES.terrace, config.terraceArea)}
                    </small>
                    <StdInput type="number" placeholder="0" value={config.terraceArea || ''} onChange={(e) => handleChange('terraceArea', parseFloat(e.target.value))} />
                </InputLabel>
                <InputLabel label="Elektrisches Garagentor-Set (Stk)" onChat={() => openChat('Garagentor')} tooltip="Sektionaltor für Standardgarage (290x240cm) inklusive elektrischem Antrieb und zwei Handsendern.">
                    <small className="text-accent text-xs mb-1 block">
                        Gesamt: {calcTotal(PRICES.garage, config.garageCount)}
                    </small>
                    <StdInput type="number" placeholder="0" value={config.garageCount || ''} onChange={(e) => handleChange('garageCount', parseFloat(e.target.value))} />
                </InputLabel>
                </div>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6">
                    <CheckboxCard 
                    checked={config.checkStairOut}
                    onChange={(v) => handleChange('checkStairOut', v)}
                    onChat={() => openChat('Außentreppe')}
                    title="Außentreppe"
                    subtitle={formatCurrency(PRICES.stair.out)}
                    tooltip="Wetterfeste Stahltreppe (feuerverzinkt) für den Zugang zu oberen Etagen oder Dachterrassen."
                    />
                    <CheckboxCard 
                    checked={config.checkStairIn}
                    onChange={(v) => handleChange('checkStairIn', v)}
                    onChat={() => openChat('Innentreppe')}
                    title="Innentreppe"
                    subtitle={formatCurrency(PRICES.stair.in)}
                    tooltip="Verbindungstreppe im Innenbereich. Stahlwangenkonstruktion mit versiegelten Holzstufen."
                    />
                </div>
            </SectionCard>
          </div>

        </div>

        {/* RIGHT COLUMN (STICKY SIDEBAR) */}
        <SummarySidebar config={config} result={result} onOpenPDF={() => setModalOpen(true)} />

      </div>

      {/* MOBILE BAR */}
      <MobileActionbar netto={result.netto} onOpen={() => setModalOpen(true)} />

    </div>
  );
}

export default App;